import os
import re
import time
from prompta.core.alphabet.events import *
from prompta.core.alphabet.verbs import VERBS


class EventLogger:

    def __init__(self, log_dir: str, mf_log_path: str):
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)
        self.log_dir = log_dir
        self.mf_log_path = mf_log_path
        self.event_log_path = None
        self.event_file = None
        self.last_events_list = []

    def reset(self, obs):
        if self.event_file is not None:
            self.event_file.close()
        time_stamp = time.strftime("%Y%m%d_%H%M%S", time.localtime())
        self.event_log_path = os.path.join(self.log_dir, f"events-{time_stamp}.log")
        self.event_file = open(self.event_log_path, "w", encoding="utf-8")
        self.event_file.close()
        self.last_obs = obs[-1][-1]
        self.last_action = None
        
    def step(self, obs, action: str=None):
        self.last_events_list.clear()
        self.event_file = open(self.event_log_path, "a", encoding="utf-8")
        self.write_action(action)
        self.write_obs(obs)
        self.last_action = action
        self.last_obs = obs[-1][-1]
        self.event_file.close()

    def write_obs(self, obs):
        obs = obs[-1][-1]
        status = obs['status']
        inventory = obs['inventory']

        # Describe the bot's current position and orientation
        position = status['position']
        velocity = status['velocity']
        yaw = status['yaw']
        pitch = status['pitch']
        biome = status['biome']
        health = status['health']
        food = status['food']
        saturation = status['saturation']
        time_of_day = status['timeOfDay']
        entities = status['entities']
        voxels = obs['voxels']

        self.last_events_list.append(f"{TimeOfDayEvent.to_symbol(time_of_day)}")
        self.last_events_list.append(f"{BiomeEvent.to_symbol(biome)}")
        self.last_events_list.append(f"{HealthEvent.to_symbol(health)}")
        self.last_events_list.append(f"{FoodEvent.to_symbol(food)}")

        for entity in entities:
            self.last_events_list.append(f"{EntityEvent.to_symbol(entity)}")

        for item, quantity in inventory.items():
            self.last_events_list.append(f"{OwnEvent.to_symbol(quantity, item)}")

        if self.last_obs is not None:
            last_inventory = self.last_obs['inventory']
            for item, quantity in inventory.items():
                last_quantity = last_inventory.get(item, 0)
                if quantity != last_quantity:
                    event = quantity - last_quantity
                else:
                    continue
                if event > 0:
                    self.last_events_list.append(f"{GaindEvent.to_symbol(event, item)}")
                else:
                    self.last_events_list.append(f"{LostEvent.to_symbol(-event, item)}")
            
            for item, last_quantity in last_inventory.items():
                if item not in inventory:
                    self.last_events_list.append(f"{LostEvent.to_symbol(last_quantity, item)}")

        for event in self.last_events_list:
            self.event_file.write(f"{event}\n")
        
    def write_action(self, action):
        if action:
            if '\n' in action:
                actions = action.split('\n')
                action = None
                for a in actions:
                    if 'await' in a:
                        action = a
                        break
            if action is None:
                return
            events = []
            for verb in VERBS:
                idtf = verb.identifier
                if idtf in action:
                    events.append(verb.to_symbol(action))
                    if "exploreUntilFind" in action:
                        with open(self.mf_log_path, "r", encoding="utf-8") as mf_log:
                            mf_log_lines = [line for line in mf_log if '[Action Event]' in line]
                        action_result = mf_log_lines[-1]
                        if verb.to_symbol(action) in action_result:
                            action_result = "succeeded" if "succeeded" in action_result else "failed"
                            events.append(f"{verb.to_symbol(action)} {action_result}")
                    self.last_events_list.extend(events)
                    for event in events:
                        self.event_file.write(f"{event}\n")
                    return
                else:
                    continue

            self.event_file.write(f"[Matching Error] {action}\n")

    def get_body_info(self, attr):
        if attr >= 15:
            return "good"
        elif attr >= 10:
            return "fair"
        elif attr >= 5:
            return "poor"
        else:
            return "agonal"
